from lxml import etree
from cryptography.hazmat.primitives import hashes, serialization
from cryptography.hazmat.primitives.asymmetric import padding
from cryptography import x509
import base64

# ----------------- CONFIG -----------------
CFDI_XML = "cfdi.xml"                   # Cambia esto por tu propio XML
XSLT_FILE = "cadenaoriginal_4_0.xslt"   # Puedes descargarlo desde el sitio web del SAT
PRIVATE_KEY_FILE = "barcodesoft.key"   # Cambia esto por tu propia clave privada
CERT_FILE = "barcodesoft.pem"          # Cambia esto por tu propio certificado
SIGNED_XML = "cfdi_signed.xml"

# ----------------- LOAD CFDI XML -----------------
parser = etree.XMLParser(remove_blank_text=True)
xml_doc = etree.parse(CFDI_XML, parser)

# ----------------- LOAD XSLT AND GENERATE CADENA ORIGINAL -----------------
xslt_doc = etree.parse(XSLT_FILE)
transform = etree.XSLT(xslt_doc)
cadena_original = str(transform(xml_doc))
print("Cadena Original:\n", cadena_original)

# ----------------- LOAD PRIVATE KEY -----------------
with open(PRIVATE_KEY_FILE, "rb") as f:
    private_key = serialization.load_pem_private_key(f.read(), password=None)

# ----------------- SIGN CADENA ORIGINAL -----------------
signature = private_key.sign(
    cadena_original.encode("utf-8"),
    padding.PKCS1v15(),
    hashes.SHA256()
)
sello_digital = base64.b64encode(signature).decode("utf-8")
print("\nSello Digital:\n", sello_digital)

# ----------------- LOAD CERTIFICATE -----------------
with open(CERT_FILE, "rb") as f:
    cert_data = f.read()
cert = x509.load_pem_x509_certificate(cert_data)

# Base64 of certificate (without headers)
cert_b64 = base64.b64encode(cert.public_bytes(serialization.Encoding.DER)).decode("utf-8")
# Serial number (NoCertificado) as string
serial_number = str(cert.serial_number)

print("\nCertificado Base64 (truncated):\n", cert_b64[:60], "...")
print("NoCertificado:", serial_number)

# ----------------- INSERT Sello, Certificado, NoCertificado -----------------
root = xml_doc.getroot()
root.set("Sello", sello_digital)
root.set("Certificado", cert_b64)
root.set("NoCertificado", serial_number)

# ----------------- WRITE SIGNED XML -----------------
xml_doc.write(SIGNED_XML, pretty_print=True, xml_declaration=True, encoding="UTF-8")
print(f"\n✔ Signed CFDI XML written to {SIGNED_XML}")
